'use client';

import { useState, useEffect, useCallback } from 'react';
import { defaultRestaurantInfo } from '@/lib/menu-data';
import type { RestaurantInfo } from '@/lib/types';

const RESTAURANT_INFO_KEY = 'restaurantInfo';

export function useRestaurantInfo() {
  const [restaurantInfo, setRestaurantInfo] = useState<RestaurantInfo>(defaultRestaurantInfo);
  const [isLoaded, setIsLoaded] = useState(false);

  useEffect(() => {
    try {
      const storedInfo = localStorage.getItem(RESTAURANT_INFO_KEY);
      if (storedInfo) {
        // Merge stored info with defaults to ensure new fields are present
        const parsedInfo = JSON.parse(storedInfo);
        setRestaurantInfo(prev => ({ ...defaultRestaurantInfo, ...parsedInfo, name: parsedInfo.name || defaultRestaurantInfo.name, carouselImages: parsedInfo.carouselImages || defaultRestaurantInfo.carouselImages }));
      }
    } catch (error) {
      console.error("Failed to parse restaurant info from localStorage", error);
      setRestaurantInfo(defaultRestaurantInfo);
    }
    setIsLoaded(true);
  }, []);

  const updateRestaurantInfo = useCallback((newInfo: Partial<RestaurantInfo>) => {
    try {
      setRestaurantInfo(prevInfo => {
        const updatedInfo = { ...prevInfo, ...newInfo };
        localStorage.setItem(RESTAURANT_INFO_KEY, JSON.stringify(updatedInfo));
        return updatedInfo;
      });
    } catch (error) {
       console.error("Failed to save restaurant info to localStorage", error);
    }
  }, []);

  return { restaurantInfo, updateRestaurantInfo, isLoaded };
}
